<?php

namespace App\Support;

use Illuminate\Contracts\Support\Arrayable;

use function PHPUnit\Framework\isNull;

class Section extends Content implements Arrayable
{
    public function __construct(
        protected string $key,
        protected array $data,
        protected ?Page $parent = null
    ) {
        //
    }

    public function key(): string
    {
        return $this->key;
    }

    /**
     * Get the parent page of the section.
     */
    public function parent(): ?Page
    {
        return $this->parent;
    }

    /**
     * Updates a specific section of the page identified by the given key.
     *
     * @param string $key The key of the section to update.
     * @param mixed $data The data to update within the section. This can be an associative array of attributes.
     * @throws InvalidArgumentException If the specified section key does not exist.
     */
    public function update(string $key, mixed $data): void
    {
        $this->parent()->patch(
            [
                $this->key() => [$key => $data],
            ],
            false
        );
    }

    /**
     * Patches multiple sections with new data.
     *
     * - If `$replace` is `false`, each section's data will be merged with the existing data.
     * - If `$replace` is `true`, the data for each section will be completely replaced, removing any existing data.
     *
     * @param array $data An associative array where the keys are section names and the values are the data to patch.
     * @param bool $replace Whether to replace the entire section data (`true`) or merge it with existing data (`false`).
     * @throws InvalidArgumentException If a section key does not exist.
     */
    public function patch(mixed $data, bool $replace = false): void
    {
        $this->parent()->update($this->key(), $data, $replace);
    }

    public function get(string $key, mixed $default = null): mixed
    {
        $value = $this->data()[$key] ?? null;
        if (!$value) {
            if (isset($default)) {
                return $default;
            }

            throw new \InvalidArgumentException("Key [{$key}] not found");
        }

        return $this->processLocalization($value);
    }

    /**
     * Get the data of the content by key without processing localization
     */
    public function getRaw(string $key, mixed $default = null): mixed
    {
        $value = $this->data()[$key] ?? null;
        if (!$value) {
            if (isset($default)) {
                return $default;
            }

            throw new \InvalidArgumentException("Key [{$key}] not found");
        }

        return $value;
    }

    public function all(): array
    {
        return array_map([$this, "processLocalization"], $this->data());
    }

    public function data(): mixed
    {
        return [...$this->data];
    }

    public function toArray(): array
    {
        return $this->all();
    }
}
