<?php

namespace App\Support;

use App\Models\SitePage;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use InvalidArgumentException;

abstract class Page extends Content
{
    protected array $data = [];
    protected SitePage $model;

    public function __construct()
    {
        $this->model = SitePage::firstOrCreate(
            ["key" => $this->key()],
            ["content" => []]
        );

        $this->data = $this->model?->content ?? [];
    }

    /**
     * Creates a new instance of the page.
     */
    public static function make(): static
    {
        return new static();
    }

    /**
     * Returns the Eloquent model associated with the page.
     */
    public function model(): SitePage
    {
        return $this->model;
    }

    public function sections(): Collection
    {
        $sections = collect();

        foreach ($this->data as $key => $section) {
            $sections->push(new Section($key, $section, $this));
        }

        return $sections;
    }

    /**
     * Updates a specific section of the page identified by the given key.
     *
     * - If `$replace` is `false`, only the specified attributes within the section will be updated, merging with existing data.
     * - If `$replace` is `null`, the section's data will be merged with the existing data.
     * - If `$replace` is `true`, the entire section's data will be replaced with the provided data, removing any existing attributes.
     * - If the section does not exist, it will be created.
     *
     * @param string $key The key of the section to update.
     * @param mixed $data The data to update within the section. This can be an associative array of attributes.
     * @param bool $replace Whether to replace the entire section's data (`true`) or merge with existing data (`false`).
     */
    public function update(
        string $key,
        mixed $data,
        ?bool $replace = null
    ): void {
        if ($replace === null) {
            $this->data[$key] = array_merge($this->data[$key] ?? [], $data);
        } else {
            $this->data[$key] = $replace
                ? $data
                : array_merge($this->data[$key] ?? [], $data);
        }

        $page = SitePage::where("key", $this->key());
        if ($page->exists()) {
            $page->update(["content" => $this->data]);
            return;
        }

        SitePage::create(["key" => $this->key(), "content" => $this->data]);
    }

    /**
     * Patches multiple sections with new data.
     *
     * - If `$replace` is `false`, each section's data will be merged with the existing data.
     * - If `$replace` is `true`, the data for each section will be completely replaced, removing any existing data.
     * - If a section does not exist, it will be created.
     *
     * @param array $data An associative array where the keys are section names and the values are the data to patch.
     * @param bool $replace Whether to replace the entire section data (`true`) or merge it with existing data (`false`).
     */
    public function patch(array $data, bool $replace = false): void
    {
        if ($replace) {
            $this->data = $data;
            SitePage::where("key", $this->key())->update(["content" => $data]);
        }

        foreach ($data as $key => $value) {
            $this->update($key, $value);
        }
    }

    public function get(string $key, mixed $default = null): Section
    {
        $value = $this->sections()->firstWhere(
            fn($section) => $section->key() === $key
        );
        if (!$value) {
            if (isset($default)) {
                return new Section($key, $default, $this);
            }

            throw new InvalidArgumentException("Section [{$key}] not found");
        }

        return $value;
    }

    public function all(): array
    {
        $all = $this->sections()
            ->mapWithKeys(fn($section) => [$section->key() => $section->all()])
            ->all();

        $all = array_map([$this, "processLocalization"], $all);

        return $all;
    }

    public function seo(): void
    {
        throw new Exception("Method not implemented");
    }
}
