<?php

namespace App\Support;

abstract class Content
{
    /**
     * Get the key of the content.
     */
    abstract public function key(): string;

    /**
     * Patches multiple sections with new data.
     *
     * - If `$replace` is `false`, each section's data will be merged with the existing data.
     * - If `$replace` is `true`, the data for each section will be completely replaced, removing any existing data.
     *
     * @param array $data An associative array where the keys are section names and the values are the data to patch.
     * @param bool $replace Whether to replace the entire section data (`true`) or merge it with existing data (`false`).
     * @throws InvalidArgumentException If a section key does not exist.
     */
    abstract public function patch(array $data, bool $replace = false): void;

    /**
     * Get the data of the content by key.
     */
    abstract public function get(string $key, mixed $default = null): mixed;

    /**
     * Get all the data of the content.
     */
    abstract public function all(): array;

    protected function isLocaleArray(array $value, array $supportedLocales): bool
    {
        $keys = array_keys($value);
        return count(array_intersect($keys, $supportedLocales)) > 0;
    }

    protected function processLocalization(mixed $value)
    {
        if (is_array($value) && (isset($value['en']) || isset($value['ar']))) {
            $locale = app()->getLocale();
            $fallbackLocale = config('app.fallbackLocale');

            return $value[$locale] ?? $value[$fallbackLocale] ?? null;
        }

        if (is_array($value)) {
            return array_map([$this, 'processLocalization'], $value);
        }

        return $value;
    }
}
