<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Request as ModelsRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Mail\NewRequestNotification;
use App\Mail\InternalNewRequestNotification;
use App\Models\OurService;
use App\Models\Package;
use App\Models\Setting;
use Carbon\Carbon;
use Illuminate\Support\Facades\Mail;

class RequestController extends Controller
{
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string',
            'email' => 'required|email',
            'phone_number' => 'required|string',
            'message' => 'required|string',
            'date' => 'nullable|date',
            'time' => 'nullable|date_format:H:i',
            'type' => 'required|in:service,package',
            'type_id' => 'required',
        ]);
        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }
        $req = new ModelsRequest();
        $req->name = $request->name;
        $req->email = $request->email;
        $req->phone_number = $request->phone_number;
        $req->type_id = $request->type_id;
        $req->date = $request->date;
        $req->time = $request->time;
        $req->message = $request->message;
        $req->type = $request->type === 'service' ? 'App\Models\OurService' : 'App\Models\Package';
        $req->save();


        if ($request->type == 'service') {
            $service = OurService::find($request->type_id);
            $typeName = $service?->getTranslation('title', 'en');
        } elseif ($request->type == 'package') {
            $package = Package::find($request->type_id);
            $typeName = $package?->getTranslation('title', 'en');
        }
        $emailData = [
            'name' => $request->name,
            'email' => $request->email,
            'phone_number' => $request->phone_number,
            'message' => $request->message,
            'date' => $request->date ? Carbon::parse($request->date)->format('F j, Y') : null,
            'time' => $request->time ? Carbon::createFromFormat('H:i', $request->time)->format('h:i A') : null,
            'type' => $request->type,
            'typeName' => $typeName,
        ];

        try {
            $clientEmail = config("mail_senders.request_client");
            $salesEmail = config("mail_senders.request_sales");

            Mail::to($salesEmail['to_email'])->send(tap(new InternalNewRequestNotification($emailData), function ($m) use ($salesEmail) {
                $m->from($salesEmail['from_email'], $salesEmail['name']);
            }));

            Mail::to($request->email)->send(
                tap(new NewRequestNotification($emailData), function ($m) use ($clientEmail) {
                    $m->from($clientEmail['from_email'], $clientEmail['name']);
                })
            );
        } catch (\Exception $e) {
            \Log::error('Mail failed: ' . $e->getMessage());
        }

        return response()->json([
            'status' => true,
            'message' => 'Request submitted successfully'
        ], 201);
    }
}
