<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ServiceRequestResource\Pages;
use App\Models\Request;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Filament\Forms;
use pxlrbt\FilamentExcel\Actions\Tables\ExportAction as ExcelExportAction;
use pxlrbt\FilamentExcel\Actions\Tables\ExportBulkAction as ExcelExportBulkAction;
use pxlrbt\FilamentExcel\Exports\ExcelExport;
use pxlrbt\FilamentExcel\Columns\Column;
use Maatwebsite\Excel\Excel as ExcelWriter;
use App\Enums\Role;

class ServiceRequestResource extends Resource
{
    protected static ?string $model = Request::class;

    protected static ?string $navigationIcon = 'heroicon-o-bookmark-square';

    public static function canViewAny(): bool
    {
        return auth()->user()?->hasRole(Role::admin->value) ?? false;
    }
    public static function canCreate(): bool
    {
        return auth()->user()?->hasRole(Role::admin->value) ?? false;
    }
    public static function canEdit($record): bool
    {
        return auth()->user()?->hasRole(Role::admin->value) ?? false;
    }
    public static function canDelete($record): bool
    {
        return auth()->user()?->hasRole(Role::admin->value) ?? false;
    }

    public static function getPluralModelLabel(): string
    {
        return __('requests.service-requests');
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()->where('type', 'App\Models\OurService')->with('requestable');
    }

    public static function form(Form $form): Form
    {
        return $form->schema([
            Forms\Components\Section::make(__("requests.details"))
                ->description(__("requests.details-description"))
                ->aside()
                ->schema([
                    Forms\Components\TextInput::make("name")
                        ->label(__("requests.name"))
                        ->maxLength(255),

                    Forms\Components\TextInput::make("email")
                        ->label(__("requests.email"))
                        ->email()
                        ->maxLength(255),

                    Forms\Components\TextInput::make("phone_number")
                        ->label(__("requests.phone-number")),

                    Forms\Components\TextInput::make('service_name')
                        ->label(__('requests.service-name'))
                        ->formatStateUsing(
                            fn($record) =>
                            optional($record->requestable)?->getTranslation('title', app()->getLocale())
                        )
                        ->disabled()
                        ->dehydrated(false),

                    Forms\Components\TextInput::make('service_price')
                        ->suffix('$')
                        ->label(__('requests.price'))
                        ->formatStateUsing(
                            fn($record) =>
                            optional($record->requestable)?->price
                        )
                        ->disabled()
                        ->dehydrated(false),

                    Forms\Components\TextInput::make("date")
                        ->label(__("requests.date")),

                    Forms\Components\TimePicker::make("time")
                        ->label(__("requests.time"))
                        ->withoutSeconds()
                        ->displayFormat('h:i A')
                        ->format('H:i:s'),

                    Forms\Components\Textarea::make("message")
                        ->label(__("requests.message"))
                ]),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('serial')
                    ->label('#')
                    ->rowIndex(),
                Tables\Columns\TextColumn::make("name")
                    ->label(__("requests.name"))
                    ->searchable(),
                Tables\Columns\TextColumn::make("email")
                    ->label(__("requests.email"))
                    ->searchable(),
                Tables\Columns\TextColumn::make("phone_number")
                    ->label(__("requests.phone-number"))
                    ->searchable(),
                Tables\Columns\TextColumn::make('service_name')
                    ->label(__('requests.service-name'))
                    ->getStateUsing(
                        fn($record) =>
                        optional($record->requestable)?->getTranslation('title', app()->getLocale())
                    )
                    ->searchable(),
                Tables\Columns\TextColumn::make('service_price')
                    ->label(__('requests.price'))
                    ->getStateUsing(
                        fn($record) =>
                        optional($record->requestable)?->price
                    )
                    ->suffix(' $')
                    ->numeric(),


                Tables\Columns\TextColumn::make("date")
                    ->label(__("requests.date"))
                    ->searchable(),
                Tables\Columns\TextColumn::make("time")
                    ->label(__("requests.time"))
                    ->searchable()
                    ->time('h:i A'),

                Tables\Columns\TextColumn::make("created_at")
                    ->label(__("requests.created_at"))
                    ->dateTime('M d, Y h:i A')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                Tables\Columns\TextColumn::make("updated_at")
                    ->label(__("requests.updated_at"))
                    ->dateTime('M d, Y h:i A')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->headerActions([
                ExcelExportAction::make('export')
                    ->label(__('Export'))
                    ->exports([
                        ExcelExport::make(__('Detailed'))
                            ->fromTable()
                            ->except(['serial'])
                            ->modifyQueryUsing(fn(Builder $query) => $query->with('requestable'))
                            ->withWriterType(ExcelWriter::XLSX)
                            ->withFilename('service-requests-' . now()->format('Y-m-d-His'))
                            ->withColumns([
                                Column::make('service_price')
                                    ->heading(__('requests.price'))
                                    ->getStateUsing(
                                        fn($record) =>
                                        optional($record->requestable)?->price . ' $'
                                    ),
                                Column::make('message')->heading(__('requests.message')),
                                Column::make('created_at')
                                    ->heading(__('requests.created_at'))
                                    ->formatStateUsing(
                                        fn($state, $record) =>
                                        optional($record->created_at)?->timezone(config('app.timezone'))->format('M d, Y h:i A')
                                    ),
                            ]),
                    ]),
            ])
            ->filters([
                //
            ])
            ->bulkActions([
                ExcelExportBulkAction::make()
                    ->label(__('Export selected'))
                    ->exports([
                        ExcelExport::make(__('Detailed'))
                            ->fromTable()
                            ->except(['serial'])
                            ->withWriterType(ExcelWriter::XLSX)
                            ->withColumns([
                                Column::make('service_price')
                                    ->heading(__('requests.price'))
                                    ->getStateUsing(
                                        fn($record) =>
                                        optional($record->requestable)?->price . ' $'
                                    ),
                                Column::make('message')->heading(__('requests.message')),
                                Column::make('created_at')
                                    ->heading(__('requests.created_at'))
                                    ->formatStateUsing(
                                        fn($state, $record) =>
                                        optional($record->created_at)?->timezone(config('app.timezone'))->format('M d, Y h:i A')
                                    ),

                            ]),
                    ]),
            ])
            ->actions([Tables\Actions\ViewAction::make()->slideOver()]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListServiceRequests::route('/'),
        ];
    }
}
