<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PackageResource\Pages;
use App\Models\Package;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

class PackageResource extends Resource
{
    protected static ?string $model = Package::class;

    protected static ?string $navigationIcon = 'heroicon-o-cube';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Grid::make(1)
                    ->schema([
                        Forms\Components\SpatieMediaLibraryFileUpload::make('image')
                            ->label(__('package.package-image-label'))
                            ->placeholder(__('package.package-image-placeholder'))
                            ->collection(\App\Models\Package::MEDIA_COLLECTION_IMAGE)
                            ->required()
                            ->columnSpanFull(),
                    ]),
                Forms\Components\TextInput::make('title')
                    ->label(__('package.title-label'))
                    ->placeholder(__('package.title-placeholder'))
                    ->required()
                    ->maxLength(255)
                    ->translatable(),
                Forms\Components\Select::make('package_type_id')
                    ->label(__('package.package-type'))
                    ->relationship('packageType', 'title')
                    ->createOptionForm([
                        Forms\Components\TextInput::make('title')
                            ->label(__('package-type.title-label'))
                            ->placeholder(__('package-type.title-placeholder'))
                            ->required()
                            ->maxLength(255)
                            ->translatable(),
                    ])
                    ->required(),
                Forms\Components\TextInput::make('validity')
                    ->label(__('package.validity'))
                    ->required()
                    ->numeric(),
                Forms\Components\Select::make('validity_type')
                    ->label(__('package.validity-type'))
                    ->options([
                        'days' => __('package.days'),
                        'months' => __('package.months'),
                        'years' => __('package.years'),
                    ])
                    ->required(),
                Forms\Components\TextInput::make('price')
                    ->label(__('package.price'))
                    ->required()
                    ->numeric(),
                Forms\Components\TextInput::make('discount_rate')
                    ->label(__('package.discount-rate-label'))
                    ->prefix('%')
                    ->placeholder(__('package.discount-rate-placeholder'))
                    ->numeric()
                    ->required(),
                Forms\Components\TextInput::make('extra_discount_rate')
                    ->label(__('package.extra-discount-label'))
                    ->prefix('%')
                    ->placeholder(__('package.extra-discount-placeholder'))
                    ->numeric()
                    ->nullable(),
                /*   Forms\Components\TextInput::make('price_before')
                    ->label(__('package.price-before'))
                    ->required()
                    ->numeric(),
                Forms\Components\TextInput::make('price_after')
                    ->label(__('package.price-after'))
                    ->required()
                    ->numeric(),*/

                Forms\Components\Grid::make(1)->schema([
                    Forms\Components\Repeater::make('packageServices')
                        ->label(__('package.services'))
                        ->relationship('packageServices')
                        ->schema([
                            Forms\Components\Select::make('our_service_id')
                                ->label(__('package.service'))
                                ->options(function (Forms\Get $get) {
                                    $items = collect($get('../../packageServices') ?? []);
                                    $selected = $items->pluck('our_service_id')->filter()->all();
                                    $current = $get('our_service_id'); // keep the current value visible
                                    $exclude = array_diff($selected, [$current]);

                                    return \App\Models\OurService::query()
                                        ->whereNotIn('id', $exclude)
                                        ->pluck('title', 'id')
                                        ->toArray();
                                })
                                ->searchable()
                                ->live() // update siblings immediately
                                ->disableOptionsWhenSelectedInSiblingRepeaterItems() // prevent duplicates in UI
                                ->distinct() // validate uniqueness across repeater items
                                ->required(),

                            Forms\Components\TextInput::make('count')
                                ->label(__('package.count'))
                                ->numeric()
                                ->default(1)
                                ->required(),
                        ])
                        ->columns(2)
                        ->columnSpanFull(),
                ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('serial')
                    ->label('#')
                    ->rowIndex(),
                Tables\Columns\SpatieMediaLibraryImageColumn::make('image')
                    ->collection(\App\Models\Package::MEDIA_COLLECTION_IMAGE)
                    ->label(__('package.package-image-label'))
                    ->width(64)
                    ->height(64)
                    ->circular(),
                Tables\Columns\TextColumn::make('title')
                    ->label(__('package.title-label'))
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('packageType.title')
                    ->label(__('package.package-type'))
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('validity')
                    ->label(__('package.validity'))
                    ->searchable()
                    ->getStateUsing(fn($record) => $record->validity . ' ' . __('package.' . $record->validity_type)),
                Tables\Columns\TextColumn::make('price')
                    ->money('SAR')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('discount_rate')
                    ->suffix('%')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('extra_discount_rate')
                    ->suffix('%')
                    ->searchable()
                    ->sortable(),
                /* Tables\Columns\TextColumn::make('price_before')
                    ->money('SAR')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('price_after')
                    ->money('SAR')
                    ->searchable()
                    ->sortable(),*/
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime('M d, Y h:i A')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime('M d, Y h:i A')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make()
                    ->icon('heroicon-o-trash')
                    ->label(__('Delete'))
                    ->requiresConfirmation()
                    ->modalHeading(__('Delete package'))
                    ->modalDescription(__('Are you sure you want to delete this package?'))
                    ->color('danger'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPackages::route('/'),
            'create' => Pages\CreatePackage::route('/create'),
            'edit' => Pages\EditPackage::route('/{record}/edit'),
        ];
    }
}
