<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PackageRequestResource\Pages;
use App\Models\Request;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use pxlrbt\FilamentExcel\Actions\Tables\ExportAction as ExcelExportAction;
use pxlrbt\FilamentExcel\Actions\Tables\ExportBulkAction as ExcelExportBulkAction;
use pxlrbt\FilamentExcel\Exports\ExcelExport;
use pxlrbt\FilamentExcel\Columns\Column;
use Maatwebsite\Excel\Excel as ExcelWriter;
use App\Enums\Role;

class PackageRequestResource extends Resource
{
    protected static ?string $model = Request::class;

    protected static ?string $navigationIcon = 'heroicon-o-queue-list';

    public static function canViewAny(): bool
    {
        return auth()->user()?->hasRole(Role::admin->value) ?? false;
    }
    public static function canCreate(): bool
    {
        return auth()->user()?->hasRole(Role::admin->value) ?? false;
    }
    public static function canEdit($record): bool
    {
        return auth()->user()?->hasRole(Role::admin->value) ?? false;
    }
    public static function canDelete($record): bool
    {
        return auth()->user()?->hasRole(Role::admin->value) ?? false;
    }


    public static function getPluralModelLabel(): string
    {
        return __('requests.package-requests');
    }


    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()->where('type', 'App\Models\Package')->with('requestable');
    }
    public static function form(Form $form): Form
    {
        return $form->schema([
            Forms\Components\Section::make(__("requests.details"))
                ->description(__("requests.details-description"))
                ->aside()
                ->schema([
                    Forms\Components\TextInput::make("name")
                        ->label(__("requests.name"))
                        ->maxLength(255),

                    Forms\Components\TextInput::make("email")
                        ->label(__("requests.email"))
                        ->email()
                        ->maxLength(255),

                    Forms\Components\TextInput::make("phone_number")
                        ->label(__("requests.phone-number")),

                    Forms\Components\TextInput::make('package_name')
                        ->label(__('requests.package-name'))
                        ->formatStateUsing(
                            fn($record) =>
                            optional($record->requestable)?->getTranslation('title', app()->getLocale())
                        )
                        ->disabled()
                        ->dehydrated(false),

                    Forms\Components\TextInput::make('package_price')
                        ->suffix('$')
                        ->label(__('requests.price'))
                        ->formatStateUsing(function ($record) {
                            $config    = \App\Support\OurServicesPage::make();
                            $vatPct    = (float) ($config->header()->get('VAT') ?? 0);
                            $base      = (float) (optional($record->requestable)?->price ?? 0);
                            $extraPct  = (float) (optional($record->requestable)?->extra_discount_rate ?? 0);

                            $afterDiscount = $base - ($base * $extraPct / 100);
                            $final         = $afterDiscount + ($afterDiscount * $vatPct / 100);
                            return round($final, 2);
                        })
                        /* ->formatStateUsing(
                            fn($record) =>
                            optional($record->requestable)?->price
                        )*/
                        ->disabled()
                        ->dehydrated(false),

                    Forms\Components\TextInput::make("date")
                        ->label(__("requests.date")),

                    Forms\Components\TimePicker::make("time")
                        ->label(__("requests.time"))
                        ->withoutSeconds()
                        ->displayFormat('h:i A')
                        ->format('H:i:s'),

                    Forms\Components\Textarea::make("message")
                        ->label(__("requests.message"))
                ]),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('serial')
                    ->label('#')
                    ->rowIndex(),
                Tables\Columns\TextColumn::make("name")
                    ->label(__("requests.name"))
                    ->searchable(),
                Tables\Columns\TextColumn::make("email")
                    ->label(__("requests.email"))
                    ->searchable(),
                Tables\Columns\TextColumn::make("phone_number")
                    ->label(__("requests.phone-number"))
                    ->searchable(),
                Tables\Columns\TextColumn::make('package_name')
                    ->label(__('requests.package-name'))
                    ->getStateUsing(
                        fn($record) =>
                        optional($record->requestable)?->getTranslation('title', app()->getLocale())
                    )
                    ->searchable(),
                Tables\Columns\TextColumn::make('package_price')
                    ->label(__('requests.price'))
                    ->state(function ($record) {
                        $config    = \App\Support\OurServicesPage::make();
                        $vatPct    = (float) ($config->header()->get('VAT') ?? 0);
                        $base      = (float) (optional($record->requestable)?->price ?? 0);
                        $extraPct  = (float) (optional($record->requestable)?->extra_discount_rate ?? 0);

                        $afterDiscount = $base - ($base * $extraPct / 100);
                        $final         = $afterDiscount + ($afterDiscount * $vatPct / 100);
                        return round($final, 2);
                    })
                    ->suffix(' $')
                    ->numeric(),


                Tables\Columns\TextColumn::make("date")
                    ->label(__("requests.date"))
                    ->searchable(),
                Tables\Columns\TextColumn::make("time")
                    ->label(__("requests.time"))
                    ->searchable()
                    ->time('h:i A'),

                Tables\Columns\TextColumn::make("created_at")
                    ->label(__("requests.created_at"))
                    ->dateTime('M d, Y h:i A')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                Tables\Columns\TextColumn::make("updated_at")
                    ->label(__("requests.updated_at"))
                    ->dateTime('M d, Y h:i A')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->headerActions([
                ExcelExportAction::make('export')
                    ->label(__('Export'))
                    ->exports([
                        ExcelExport::make(__('Detailed'))
                            ->fromTable()
                            ->except(['serial'])
                            ->modifyQueryUsing(fn(Builder $query) => $query->with('requestable'))
                            ->withWriterType(ExcelWriter::XLSX)
                            ->withFilename('package-requests-' . now()->format('Y-m-d-His'))
                            ->withColumns([
                                Column::make('package_price')
                                    ->heading(__('requests.price'))
                                    ->getStateUsing(function ($record) {
                                        $config    = \App\Support\OurServicesPage::make();
                                        $vatPct    = (float) ($config->header()->get('VAT') ?? 0);
                                        $base      = (float) (optional($record->requestable)?->price ?? 0);
                                        $extraPct  = (float) (optional($record->requestable)?->extra_discount_rate ?? 0);

                                        $afterDiscount = $base - ($base * $extraPct / 100);
                                        $final         = $afterDiscount + ($afterDiscount * $vatPct / 100);
                                        return round($final, 2) . ' $';
                                    }),
                                /* ->getStateUsing(
                                        fn($record) =>
                                        optional($record->requestable)?->price . ' $'
                                    ),*/
                                Column::make('message')->heading(__('requests.message')),
                                Column::make('created_at')
                                    ->heading(__('requests.created_at'))
                                    ->formatStateUsing(
                                        fn($state, $record) =>
                                        optional($record->created_at)?->timezone(config('app.timezone'))->format('M d, Y h:i A')
                                    ),
                                /*  Column::make('name')->heading(__('requests.name')),
                                Column::make('email')->heading(__('requests.email')),
                                Column::make('phone_number')->heading(__('requests.phone-number')),
                                Column::make('package_name')
                                    ->heading(__('requests.package-name'))
                                    ->getStateUsing(
                                        fn($record) =>
                                        optional($record->requestable)?->getTranslation('title', app()->getLocale())
                                    ),
                                Column::make('package_price')
                                    ->heading(__('requests.price'))
                                    ->getStateUsing(
                                        fn($record) =>
                                        optional($record->requestable)?->price. ' $'
                                    ),
                                Column::make('date')->heading(__('requests.date')),

                                Column::make('time')
                                    ->heading(__('requests.time'))
                                    ->formatStateUsing(
                                        fn($state, $record) =>
                                        $record->time
                                            ? \Carbon\Carbon::createFromFormat('H:i:s', $record->time)->format('h:i A')
                                            : null
                                    ),

                                Column::make('message')->heading(__('requests.message')),

                                Column::make('created_at')
                                    ->heading(__('requests.created_at'))
                                    ->formatStateUsing(
                                        fn($state, $record) =>
                                        optional($record->created_at)?->timezone(config('app.timezone'))->format('M d, Y h:i A')
                                    ),*/
                            ]),
                    ]),
            ])
            ->filters([
                //
            ])
            ->bulkActions([
                ExcelExportBulkAction::make()
                    ->label(__('Export selected'))
                    ->exports([
                        ExcelExport::make(__('Detailed'))
                            ->fromTable()
                            ->except(['serial'])
                            ->withWriterType(ExcelWriter::XLSX)
                            ->withColumns([
                                Column::make('package_price')
                                    ->heading(__('requests.price'))
                                    ->getStateUsing(function ($record) {
                                        $config    = \App\Support\OurServicesPage::make();
                                        $vatPct    = (float) ($config->header()->get('VAT') ?? 0);
                                        $base      = (float) (optional($record->requestable)?->price ?? 0);
                                        $extraPct  = (float) (optional($record->requestable)?->extra_discount_rate ?? 0);

                                        $afterDiscount = $base - ($base * $extraPct / 100);
                                        $final         = $afterDiscount + ($afterDiscount * $vatPct / 100);
                                        return round($final, 2) . ' $';
                                    }),
                                /*  ->getStateUsing(
                                        fn($record) =>
                                        optional($record->requestable)?->price . ' $'
                                    ),*/
                                Column::make('message')->heading(__('requests.message')),
                                Column::make('created_at')
                                    ->heading(__('requests.created_at'))
                                    ->formatStateUsing(
                                        fn($state, $record) =>
                                        optional($record->created_at)?->timezone(config('app.timezone'))->format('M d, Y h:i A')
                                    ),
                                /*  Column::make('name')->heading(__('requests.name')),
                                Column::make('email')->heading(__('requests.email')),
                                Column::make('phone_number')->heading(__('requests.phone-number')),
                                Column::make('package_name')
                                    ->heading(__('requests.package-name'))
                                    ->getStateUsing(
                                        fn($record) =>
                                        optional($record->requestable)?->getTranslation('title', app()->getLocale())
                                    ),
                                Column::make('package_price')
                                    ->heading(__('requests.price'))
                                    ->getStateUsing(
                                        fn($record) =>
                                        optional($record->requestable)?->price. ' $'
                                    ),
                                Column::make('date')->heading(__('requests.date')),

                                Column::make('time')
                                    ->heading(__('requests.time'))
                                    ->formatStateUsing(
                                        fn($state, $record) =>
                                        $record->time
                                            ? \Carbon\Carbon::createFromFormat('H:i:s', $record->time)->format('h:i A')
                                            : null
                                    ),
                                Column::make('message')->heading(__('requests.message')),
                                Column::make('created_at')
                                    ->heading(__('requests.created_at'))
                                    ->formatStateUsing(
                                        fn($state, $record) =>
                                        optional($record->created_at)?->timezone(config('app.timezone'))->format('M d, Y h:i A')
                                    ),*/
                            ]),
                    ]),
            ])
            ->actions([Tables\Actions\ViewAction::make()->slideOver()]);
        /*->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);*/
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPackageRequests::route('/'),
        ];
    }
}
