<?php

namespace App\Filament\Resources;

use App\Filament\Resources\OurServiceResource\Pages;
use App\Models\OurService;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

class OurServiceResource extends Resource
{
    protected static ?string $model = OurService::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function form(Form $form): Form
    {
        return $form->schema([
            Forms\Components\Section::make(__('our-services.header'))
                ->description(__('our-services.header-description'))
                ->aside()
                ->schema([
                    Forms\Components\SpatieMediaLibraryFileUpload::make('image')
                        ->label(__('our-services.service-image-label'))
                        ->placeholder(__('our-services.service-image-placeholder'))
                        ->collection(\App\Models\OurService::MEDIA_COLLECTION_IMAGE)
                        ->required(),
                    Forms\Components\TextInput::make('title')
                        ->label(__('our-services.title-label'))
                        ->placeholder(__('our-services.title-placeholder'))
                        ->required()
                        ->maxLength(255)
                        ->translatable(),
                    Forms\Components\TextInput::make('unit')
                        ->label(__('our-services.unit-label'))
                        ->placeholder(__('our-services.unit-placeholder'))
                        ->required()
                        ->maxLength(255)
                        ->translatable(),
                    Forms\Components\TextInput::make('price')
                        ->prefix('SAR')
                        ->label(__('our-services.price-label'))
                        ->placeholder(__('our-services.price-placeholder'))
                        ->required()
                        ->numeric()
                        ->minValue(0),
                    //  ->step(0.01),
                    Forms\Components\RichEditor::make('description')
                        ->label(__('our-services.description-label'))
                        ->placeholder(__('our-services.description-placeholder'))
                        ->required()
                        ->columnSpanFull()
                        ->translatable(),
                ]),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('serial')
                    ->label('#')
                    ->rowIndex(),
                Tables\Columns\SpatieMediaLibraryImageColumn::make('image')
                    ->collection(\App\Models\OurService::MEDIA_COLLECTION_IMAGE)
                    ->label(__('our-services.service-image-label'))
                    ->width(64)
                    ->height(64)
                    ->circular(),
                Tables\Columns\TextColumn::make('title')
                    ->label(__('our-services.title-label'))
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('description')
                    ->label(__('our-services.description-label'))
                    ->formatStateUsing(fn($state) => strip_tags($state))
                    ->limit(50)
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('unit')
                    ->label(__('our-services.unit-label'))
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('price')
                    ->money('SAR')
                    ->label(__('our-services.price-label'))
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->label(__('our-services.created-at-label'))
                    ->dateTime('M d, Y h:i A')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->label(__('our-services.updated-at-label'))
                    ->dateTime('M d, Y h:i A')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make()
                    ->icon('heroicon-o-trash')
                    ->label(__('Delete'))
                    ->requiresConfirmation()
                    ->modalHeading(__('Delete service'))
                    ->modalDescription(__('Are you sure you want to delete this service?'))
                    ->color('danger'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListOurServices::route('/'),
            'create' => Pages\CreateOurService::route('/create'),
            'edit' => Pages\EditOurService::route('/{record}/edit'),
        ];
    }
}
