<?php

namespace App\Filament\Resources;

use App\Filament\Resources\AdminUserResource\Pages;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use App\Enums\Role;
use App\Models\User;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Validation\Rules\Password as PasswordRule;


class AdminUserResource extends Resource
{
    protected static ?string $model = User::class;

    protected static ?string $navigationIcon  = 'heroicon-o-shield-check';

    public static function shouldRegisterNavigation(): bool
    {
        return auth()->user()?->hasRole(Role::admin->value) ?? false;
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()->whereHas('roles', function ($q) {
            $q->whereIn('name', [Role::admin->value, Role::ContentCreator->value]);
        });
    }

    public static function canViewAny(): bool
    {
        return auth()->user()?->hasRole(Role::admin->value) ?? false;
    }
    public static function canCreate(): bool
    {
        return auth()->user()?->hasRole(Role::admin->value) ?? false;
    }
    public static function canEdit($record): bool
    {
        return auth()->user()?->hasRole(Role::admin->value) ?? false;
    }
    public static function canDelete($record): bool
    {
        return auth()->user()?->hasRole(Role::admin->value) ?? false;
    }

    public static function getPluralModelLabel(): string
    {
        return __('users.admin-users');
    }


    public static function form(Form $form): Form
    {
        return $form->schema([
            Forms\Components\TextInput::make('name')
                ->label(__('users.name'))
                ->required()
                ->maxLength(191),

            Forms\Components\TextInput::make('email')
                ->email()
                ->unique(ignoreRecord: true)
                ->required(),

            Forms\Components\TextInput::make('password')
                ->label(__('users.password'))
                ->password()
                ->revealable()
                ->rule(PasswordRule::defaults())
                ->required(fn(string $context) => $context === 'create')
                ->dehydrated(fn($state) => filled($state)),

            Forms\Components\Select::make('role')
                ->label(__('users.role'))
                ->options([
                    \App\Enums\Role::admin->value          => \App\Enums\Role::admin->label(),
                    \App\Enums\Role::ContentCreator->value => \App\Enums\Role::ContentCreator->label(),
                ])
                ->required()
                ->native(false)
                ->helperText(__('users.select_admin_or_content_creator'))
        ])->columns(1);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')->searchable(),
                Tables\Columns\TextColumn::make('email')->searchable(),
                Tables\Columns\TextColumn::make('roles.name')
                    ->label(__('users.role'))
                    ->badge()
                    ->formatStateUsing(function ($state) {
                        return in_array($state, [
                            \App\Enums\Role::admin->value,
                            \App\Enums\Role::ContentCreator->value,
                        ], true)
                            ? \App\Enums\Role::from($state)->label()
                            : $state;
                    }),
                Tables\Columns\TextColumn::make('created_at')->dateTime(),
            ])
            ->filters([
                SelectFilter::make('role')
                    ->label(__('users.role'))
                    ->relationship('roles', 'name')
                    ->options([
                        \App\Enums\Role::admin->value          => \App\Enums\Role::admin->label(),
                        \App\Enums\Role::ContentCreator->value => \App\Enums\Role::ContentCreator->label(),
                    ]),
            ])
            ->actions([
                Tables\Actions\EditAction::make()
                    ->visible(fn($record) => auth()->user()?->hasRole(Role::admin->value)),

                Tables\Actions\DeleteAction::make()
                    ->visible(
                        fn($record) =>
                        auth()->user()?->hasRole(Role::admin->value) && $record->isNot(auth()->user())
                    ),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make()
                    ->visible(fn() => auth()->user()?->hasRole(Role::admin->value)),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListAdminUsers::route('/'),
            'create' => Pages\CreateAdminUser::route('/create'),
            'edit' => Pages\EditAdminUser::route('/{record}/edit'),
        ];
    }
}
