<?php

namespace App\Filament\Pages;

use App\Support\WhoWeArePage;
use Exception;
use Filament\Forms\Form;
use Filament\Forms;
use Filament\Pages\Page;
use Illuminate\Contracts\Support\Htmlable;
use Filament\Actions\Action;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Rules\MaxPlainTextLength;

class WhoWeAre extends Page
{
    protected static ?string $navigationIcon = "heroicon-o-question-mark-circle";

    protected static string $view = "filament.pages.who-we-are";

    public static function canAccess(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }

    public static function shouldRegisterNavigation(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }

    public array $data = [];

    public static function getNavigationGroup(): ?string
    {
        return __("navigation.cms");
    }

    public function getTitle(): string|Htmlable
    {
        return __("who-we-are.who-we-are");
    }

    public static function getNavigationLabel(): string
    {
        return __("who-we-are.who-we-are");
    }

    public function mount(): void
    {

        $page = WhoWeArePage::make();

        $this->form->fill([
            "title" => $page->header()->getRaw("title"),
            "description" => $page->header()->getRaw("description"),
            "individual_hospitality_subtitle" => $page->individualHospitality()->getRaw("individual_hospitality_subtitle"),
            "individual_hospitality_description" => $page->individualHospitality()->getRaw("individual_hospitality_description"),
            "why_we_subtitle" => $page->whyWe()->getRaw("why_we_subtitle"),
            "why_we_description" => $page->whyWe()->getRaw("why_we_description"),
            "package_pricing_subtitle" => $page->packagePricing()->getRaw("package_pricing_subtitle"),
            "package_pricing_description" => $page->packagePricing()->getRaw("package_pricing_description"),
        ]);
    }

    public function form(Form $form): Form
    {
        return $form->statePath("data")->schema([
            Forms\Components\Section::make(strip_tags(WhoWeArePage::make()->header()->getRaw('title')[app()->getLocale()]))
                ->description(strip_tags(WhoWeArePage::make()->header()->getRaw('description')[app()->getLocale()]))
                ->aside()
                ->schema([
                    Forms\Components\SpatieMediaLibraryFileUpload::make("cover")
                        ->label(__("who-we-are.cover-label"))
                        ->collection(WhoWeArePage::MEDIA_COLLECTION_COVER)
                        ->model(WhoWeArePage::make()->model())
                        ->required(),
                    Forms\Components\TextInput::make("title")
                        ->label(__("who-we-are.title-label"))
                        ->placeholder(__("who-we-are.title-placeholder"))
                        ->required()
                        ->maxLength(20)
                        ->helperText(__('who-we-are.title-helper'))
                        ->translatable(),
                    Forms\Components\RichEditor::make("description")
                        ->label(__("who-we-are.description-label"))
                        ->placeholder(__("who-we-are.description-placeholder"))
                        ->required()
                        ->maxLength(60)
                        ->helperText(__('who-we-are.description-helper'))
                        ->translatable()
                ]),


            Forms\Components\Section::make(strip_tags(WhoWeArePage::make()->individualHospitality()->getRaw('individual_hospitality_subtitle')[app()->getLocale()]))
                ->description(strip_tags(WhoWeArePage::make()->individualHospitality()->getRaw('individual_hospitality_description')[app()->getLocale()]))
                ->aside()
                ->schema([
                    Forms\Components\TextInput::make("individual_hospitality_subtitle")
                        ->label(__("who-we-are.subtitle-label"))
                        ->placeholder(__("who-we-are.subtitle-placeholder"))
                        ->required()
                        ->maxLength(60)
                        ->helperText(__('who-we-are.subtitle-helper1'))
                        ->translatable(),
                    Forms\Components\RichEditor::make("individual_hospitality_description")
                        ->label(__("who-we-are.description-label"))
                        ->placeholder(__("who-we-are.description-placeholder"))
                        ->required()
                        ->maxLength(200)
                        ->helperText(__('who-we-are.subdescription-helper1'))
                        ->translatable()
                ]),


            Forms\Components\Section::make(strip_tags(WhoWeArePage::make()->whyWe()->getRaw('why_we_subtitle')[app()->getLocale()]))
                ->description(strip_tags(WhoWeArePage::make()->whyWe()->getRaw('why_we_description')[app()->getLocale()]))
                ->aside()
                ->schema([
                    Forms\Components\TextInput::make("why_we_subtitle")
                        ->label(__("who-we-are.subtitle-label"))
                        ->placeholder(__("who-we-are.subtitle-placeholder"))
                        ->required()
                        ->maxLength(100)
                        ->helperText(__('who-we-are.subtitle-helper2'))
                        ->translatable(),
                    Forms\Components\RichEditor::make("why_we_description")
                        ->label(__("who-we-are.description-label"))
                        ->placeholder(__("who-we-are.description-placeholder"))
                        ->required()
                        ->maxLength(200)
                        ->helperText(__('who-we-are.subdescription-helper2'))
                        ->translatable()
                ]),

            Forms\Components\Section::make(strip_tags(WhoWeArePage::make()->packagePricing()->getRaw('package_pricing_subtitle')[app()->getLocale()]))
                ->description(strip_tags(WhoWeArePage::make()->packagePricing()->getRaw('package_pricing_description')[app()->getLocale()]))
                ->aside()
                ->schema([
                    Forms\Components\TextInput::make("package_pricing_subtitle")
                        ->label(__("who-we-are.subtitle-label"))
                        ->placeholder(__("who-we-are.subtitle-placeholder"))
                        ->required()
                        ->maxLength(255)
                        ->translatable(),
                    Forms\Components\RichEditor::make("package_pricing_description")
                        ->label(__("who-we-are.description-label"))
                        ->placeholder(__("who-we-are.description-placeholder"))
                        ->required()
                        ->translatable()
                ]),

        ]);
    }

    protected function getActions(): array
    {
        return [
            Action::make(__("app.update"))
                ->icon("heroicon-o-arrow-path")
                ->action(fn() => $this->publish()),
        ];
    }

    public function publish(): void
    {
        $this->validate($this->getValidationRules());
        try {
            DB::beginTransaction();

            $page = WhoWeArePage::make();

            $image = collect($this->data["cover"])->first();
            if (
                !is_string($image) &&
                TemporaryUploadedFile::class === get_class($image)
            ) {
                $page->setCover($image);
            }
            $page->header()->patch([
                "title" => $this->data["title"],
                "description" => $this->data["description"],
            ]);
            $page->individualHospitality()->patch([
                "individual_hospitality_subtitle" => $this->data["individual_hospitality_subtitle"],
                "individual_hospitality_description" => $this->data["individual_hospitality_description"],
            ]);
            $page->whyWe()->patch([
                "why_we_subtitle" => $this->data["why_we_subtitle"],
                "why_we_description" => $this->data["why_we_description"],
            ]);
            $page->packagePricing()->patch([
                "package_pricing_subtitle" => $this->data["package_pricing_subtitle"],
                "package_pricing_description" => $this->data["package_pricing_description"],
            ]);

            Notification::make()
                ->success()
                ->title(__("app.content-update-success"))
                ->send();
            DB::commit();
        } catch (Exception $e) {
            DB::rollback();
            Log::error("Failed to update Who We Are data", [
                "message" => $e->getMessage(),
                "trace" => $e->getTraceAsString(),
            ]);

            Notification::make()
                ->danger()
                ->title(__("app.content-update-failure"))
                ->send();
        }
    }

    protected function getValidationRules(): array
    {
        $rules = [
            'data.title' => ['required', 'array'],
            'data.description' => ['required', 'array'],
            'data.individual_hospitality_subtitle' => ['required', 'array'],
            'data.individual_hospitality_description' => ['required', 'array'],
            'data.why_we_subtitle' => ['required', 'array'],
            'data.why_we_description' => ['required', 'array'],
            'data.package_pricing_subtitle' => ['required', 'array'],
            'data.package_pricing_description' => ['required', 'array'],
            'data.cover' => ['required'],
        ];

        $plainTextLimits = [
            'title' => 20,
            'description' => 60,
            'individual_hospitality_description' => 200,
            'individual_hospitality_subtitle' => 60,
            'why_we_subtitle' => 100,
            'why_we_description' => 200,
        ];

        foreach ($plainTextLimits as $field => $limit) {
            foreach ($this->data[$field] ?? [] as $locale => $_) {
                $rules["data.{$field}.{$locale}"][] = 'required';
                $rules["data.{$field}.{$locale}"][] = new MaxPlainTextLength($limit);
            }
        }

        return $rules;
    }
}
