<?php

namespace App\Filament\Pages;

use App\Support\TermsAndConditionsPage;
use Exception;
use Filament\Forms\Form;
use Filament\Forms;
use Filament\Pages\Page;
use Illuminate\Contracts\Support\Htmlable;
use Filament\Actions\Action;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;

class TermsAndConditions extends Page
{
    protected static ?string $navigationIcon = 'heroicon-o-list-bullet';

    protected static string $view = 'filament.pages.terms-and-conditions';

    public static function canAccess(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }

    public static function shouldRegisterNavigation(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }

    public array $data = [];

    public static function getNavigationGroup(): ?string
    {
        return __('navigation.cms');
    }

    public function getTitle(): string|Htmlable
    {
        return __('terms-and-conditions.terms-and-conditions');
    }

    public static function getNavigationLabel(): string
    {
        return __('terms-and-conditions.terms-of-use');
    }

    public function mount(): void
    {
        $page = TermsAndConditionsPage::make();

        $this->form->fill([


            'content' => $page->termsAndConditions()->getRaw('content'),
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->statePath('data')
            ->schema([
                Forms\Components\Section::make(__('terms-and-conditions.terms-and-conditions'))
                    ->description(__('terms-and-conditions.terms-and-conditions-description'))
                    ->aside()
                    ->schema([

                        Forms\Components\SpatieMediaLibraryFileUpload::make('cover')
                            ->label(__('terms-and-conditions.cover-label'))
                            ->collection(TermsAndConditionsPage::MEDIA_COLLECTION_COVER)
                            ->model(TermsAndConditionsPage::make()->model())
                            ->required(),

                        Forms\Components\RichEditor::make('content')
                            ->label(__('terms-and-conditions.content-label'))
                            ->placeholder(__('terms-and-conditions.content-placeholder'))
                            ->required()
                            ->translatable(),
                    ]),
            ]);
    }

    protected function getActions(): array
    {
        return [
            Action::make(__('app.update'))
                ->icon('heroicon-o-arrow-path')
                ->action(fn() => $this->publish()),
        ];
    }

    public function publish(): void
    {
        $this->validate();

        try {
            DB::beginTransaction();

            $page = TermsAndConditionsPage::make();

            $image = collect($this->data['cover'])->first();
            if (! is_string($image) && TemporaryUploadedFile::class === get_class($image)) {
                $page->setCover($image);
            }

            $page->termsAndConditions()->patch([
                'content' => $this->data['content'],
            ]);

            Notification::make()
                ->success()
                ->title(__('app.content-update-success'))
                ->send();
            DB::commit();
        } catch (Exception $e) {
            DB::rollback();
            Log::error('Failed to update Terms And Conditions data', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            Notification::make()
                ->danger()
                ->title(__('app.content-update-failure'))
                ->send();
        }
    }
}
