<?php

namespace App\Filament\Pages;

use App\Support\ContactUsPage;
use App\Support\StatisticsPage;
use Exception;
use Filament\Forms\Form;
use Filament\Forms;
use Filament\Pages\Page;
use Illuminate\Contracts\Support\Htmlable;
use Filament\Actions\Action;
use Filament\Notifications\Notification;
use Filament\Widgets\StatsOverviewWidget\Stat;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;

class Statistics extends Page
{
    protected static ?string $navigationIcon = 'heroicon-o-chart-bar-square';

    protected static string $view = 'filament.pages.statistics';

    public static function canAccess(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }

    public static function shouldRegisterNavigation(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }

    public array $data = [];

    public static function getNavigationGroup(): ?string
    {
        return __('navigation.cms');
    }

    public function getTitle(): string|Htmlable
    {
        return __('statistics.statistics');
    }

    public static function getNavigationLabel(): string
    {
        return __('statistics.statistics');
    }

    public function mount(): void
    {
        $page = StatisticsPage::make();

        $this->form->fill([
            'completed_booking_title' => $page->completedBooking()->getRaw('title'),
            'completed_booking_count' => $page->completedBooking()->getRaw('count'),
            'happy_clients_title' => $page->happyClients()->getRaw('title'),
            'happy_clients_count' => $page->happyClients()->getRaw('count'),
            'expert_team_title' => $page->expertTeam()->getRaw('title'),
            'expert_team_count' => $page->expertTeam()->getRaw('count'),
            'awards_won_title' => $page->awardsWon()->getRaw('title'),
            'awards_won_count' => $page->awardsWon()->getRaw('count'),
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->statePath('data')
            ->schema([
                Forms\Components\Section::make(StatisticsPage::make()->completedBooking()->getRaw('title')[app()->getLocale()])
                    ->description(StatisticsPage::make()->completedBooking()->getRaw('count'))
                    ->aside()
                    ->schema([
                        Forms\Components\TextInput::make('completed_booking_title')
                            ->label(__('statistics.title-label'))
                            ->placeholder(__('statistics.title-placeholder'))
                            ->required()
                            ->maxLength(255)
                            ->translatable(),

                        Forms\Components\TextInput::make('completed_booking_count')
                            ->label(__('statistics.count-label'))
                            ->placeholder(__('statistics.count-placeholder'))
                            ->required(),

                        Forms\Components\SpatieMediaLibraryFileUpload::make('completed_booking_icon')
                            ->label(__('statistics.completed-booking-icon-label'))
                            ->collection(StatisticsPage::MEDIA_COLLECTION_COMPLETED_BOOKING_ICON)
                            ->model(StatisticsPage::make()->model())
                            ->required(),
                    ]),

                Forms\Components\Section::make(StatisticsPage::make()->happyClients()->getRaw('title')[app()->getLocale()])
                    ->description(StatisticsPage::make()->happyClients()->getRaw('count'))
                    ->aside()
                    ->schema([
                        Forms\Components\TextInput::make('happy_clients_title')
                            ->label(__('statistics.title-label'))
                            ->placeholder(__('statistics.title-placeholder'))
                            ->required()
                            ->maxLength(255)
                            ->translatable(),

                        Forms\Components\TextInput::make('happy_clients_count')
                            ->label(__('statistics.count-label'))
                            ->placeholder(__('statistics.count-placeholder'))
                            ->required(),

                        Forms\Components\SpatieMediaLibraryFileUpload::make('happy_clients_icon')
                            ->label(__('statistics.happy-clients-icon-label'))
                            ->collection(StatisticsPage::MEDIA_COLLECTION_HAPPY_CLIENTS_ICON)
                            ->model(StatisticsPage::make()->model())
                            ->required(),
                    ]),

                Forms\Components\Section::make(StatisticsPage::make()->expertTeam()->getRaw('title')[app()->getLocale()])->description(StatisticsPage::make()->expertTeam()->getRaw('count'))
                    ->aside()
                    ->schema([
                        Forms\Components\TextInput::make('expert_team_title')
                            ->label(__('statistics.title-label'))
                            ->placeholder(__('statistics.title-placeholder'))
                            ->required()
                            ->maxLength(255)
                            ->translatable(),

                        Forms\Components\TextInput::make('expert_team_count')
                            ->label(__('statistics.count-label'))->placeholder(__('statistics.count-placeholder'))
                            ->required(),

                        Forms\Components\SpatieMediaLibraryFileUpload::make('expert_team_icon')
                            ->label(__('statistics.expert-team-icon-label'))
                            ->collection(StatisticsPage::MEDIA_COLLECTION_EXPERT_TEAM_ICON)
                            ->model(StatisticsPage::make()->model())
                            ->required(),
                    ]),

                Forms\Components\Section::make(StatisticsPage::make()->awardsWon()->getRaw('title')[app()->getLocale()])->description(StatisticsPage::make()->awardsWon()->getRaw('count'))
                    ->aside()
                    ->schema([
                        Forms\Components\TextInput::make('awards_won_title')
                            ->label(__('statistics.title-label'))
                            ->placeholder(__('statistics.title-placeholder'))
                            ->required()
                            ->maxLength(255)
                            ->translatable(),

                        Forms\Components\TextInput::make('awards_won_count')
                            ->label(__('statistics.count-label'))->placeholder(__('statistics.count-placeholder'))
                            ->required(),

                        Forms\Components\SpatieMediaLibraryFileUpload::make('awards_won_icon')
                            ->label(__('statistics.awards-won-icon-label'))
                            ->collection(StatisticsPage::MEDIA_COLLECTION_AWARDS_WON_ICON)
                            ->model(StatisticsPage::make()->model())
                            ->required(),
                    ]),
            ]);
    }

    protected function getActions(): array
    {
        return [
            Action::make(__('app.update'))
                ->icon('heroicon-o-arrow-path')
                ->action(fn() => $this->publish()),
        ];
    }

    public function publish(): void
    {
        $this->validate();

        try {
            DB::beginTransaction();

            $page = StatisticsPage::make();


            $image = collect($this->data['completed_booking_icon'])->first();
            if (! is_string($image) && TemporaryUploadedFile::class === get_class($image)) {
                $page->setCompletedBookingIcon($image);
            }


            $image = collect($this->data['happy_clients_icon'])->first();
            if (! is_string($image) && TemporaryUploadedFile::class === get_class($image)) {
                $page->setHappyClientsIcon($image);
            }


            $image = collect($this->data['expert_team_icon'])->first();
            if (! is_string($image) && TemporaryUploadedFile::class === get_class($image)) {
                $page->setExpertTeamIcon($image);
            }
            $image = collect($this->data['awards_won_icon'])->first();
            if (! is_string($image) && TemporaryUploadedFile::class === get_class($image)) {
                $page->setAwardsWonIcon($image);
            }

            $page->header()->patch([]);

            $page->completedBooking()->patch([
                'title' => $this->data['completed_booking_title'],
                'count' => $this->data['completed_booking_count'],
            ]);
            $page->happyClients()->patch([
                'title' => $this->data['happy_clients_title'],
                'count' => $this->data['happy_clients_count'],
            ]);
            $page->expertTeam()->patch([
                'title' => $this->data['expert_team_title'],
                'count' => $this->data['expert_team_count'],
            ]);
            $page->awardsWon()->patch([
                'title' => $this->data['awards_won_title'],
                'count' => $this->data['awards_won_count'],
            ]);

            Notification::make()
                ->success()
                ->title(__('app.content-update-success'))
                ->send();
            DB::commit();
        } catch (Exception $e) {
            DB::rollback();
            Log::error('Failed to update Statistics data', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            Notification::make()
                ->danger()
                ->title(__('app.content-update-failure'))
                ->send();
        }
    }
}
