<?php

namespace App\Filament\Pages;

use App\Models\NewsletterSubscriber;
use Exception;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Pages\Page;
use Illuminate\Contracts\Support\Htmlable;
use Filament\Actions\Action;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Mail;
use Filament\Forms\Components\FileUpload;
use Illuminate\Support\Facades\Storage;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;

class SendNewsletter extends Page
{
    protected static ?string $navigationIcon = "heroicon-o-paper-airplane";

    protected static string $view = "filament.pages.send-newsletter";

    public static function canAccess(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }

    public static function shouldRegisterNavigation(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }

    public array $data = [];

    public static function getNavigationGroup(): ?string
    {
        return __("navigation.cms");
    }

    public function getTitle(): string|Htmlable
    {
        return __("newsletter-subscribers-resource.send-newsletter-title");
    }

    public static function getNavigationLabel(): string
    {
        return __("newsletter-subscribers-resource.send-newsletter");
    }

    public function form(Form $form): Form
    {
        return $form->statePath("data")->schema([
            Forms\Components\Section::make(
                __("newsletter-subscribers-resource.send-newsletter-title")
            )
                ->description(
                    __(
                        "newsletter-subscribers-resource.send-newsletter-description"
                    )
                )
                ->aside()
                ->schema([
                    Forms\Components\TextInput::make("subject")
                        ->label(
                            __("newsletter-subscribers-resource.subject-label")
                        )
                        ->placeholder(
                            __(
                                "newsletter-subscribers-resource.subject-placeholder"
                            )
                        )
                        ->required()
                        ->maxLength(255),
                    Forms\Components\Textarea::make("body")
                        ->label(
                            __("newsletter-subscribers-resource.body-label")
                        )
                        ->placeholder(
                            __(
                                "newsletter-subscribers-resource.body-placeholder"
                            )
                        )
                        ->required()
                        ->rows(8),
                    FileUpload::make('image')
                        ->label(__('newsletter-subscribers-resource.image-label'))
                        ->image()
                        ->disk('public')
                        ->directory('newsletters')
                        ->visibility('public')
                        ->multiple(false)
                        ->helperText(__('newsletter-subscribers-resource.image-helper'))
                        ->nullable()
                ]),
        ]);
    }

    protected function getActions(): array
    {
        return [
            Action::make("back")
                ->label(__("newsletter-subscribers-resource.back"))
                ->icon(
                    app()->getLocale() === "en" ? "heroicon-o-arrow-left" : null
                )
                ->url(
                    route(
                        "filament.admin.resources.newsletter-subscribers.index"
                    )
                ),
            Action::make(__("newsletter-subscribers-resource.send"))
                ->icon(
                    app()->getLocale() === "en"
                        ? "heroicon-o-paper-airplane"
                        : null
                )
                ->action(fn() => $this->sendNewsletter()),
        ];
    }

    public function sendNewsletter(): void
    {
        $this->validate();

        $subject = $this->data["subject"] ?? "";
        $body = $this->data["body"] ?? "";
        $image = $this->data['image'] ?? null;
        if (is_array($image)) {
            $image = reset($image);
        }

        $imageUrl = null;

        if ($image instanceof TemporaryUploadedFile) {
            $path = $image->store('newsletters', 'public');
            $imageUrl = Storage::disk('public')->url($path);
        } elseif (is_string($image)) {
            $source = storage_path('app/public/' . $image);
            if (is_file($source)) {
                $destDir  = 'newsletters';
                $filename = pathinfo($source, PATHINFO_FILENAME) . '.jpg';
                $destRel  = $destDir . '/' . $filename;
                Storage::disk('public')->put($destRel, file_get_contents($source));
                $imageUrl = Storage::disk('public')->url($destRel);
            }
        }
        try {
            $subscribers = NewsletterSubscriber::pluck("email")->toArray();
            $infoEmail = config("mail_senders.newsletter");

            /*foreach ($subscribers as $email) {
                Mail::raw($body, function ($message) use ($email, $subject) {
                    $message->to($email)->subject($subject);
                });
            }*/

            foreach ($subscribers as $email) {
                Mail::send('emails.newsletter', [
                    'subject'  => $subject,
                    'body'     => $body,
                ], function ($message) use ($email, $subject, $imageUrl, $infoEmail) {
                    $message->to($email)
                    ->from($infoEmail['from_email'], $infoEmail['name'])
                    ->subject($subject);
                    if ($imageUrl) {
                        $relativePath = str_replace(Storage::disk('public')->url(''), '', $imageUrl);
                        $fullPath = Storage::disk('public')->path($relativePath);

                        if (file_exists($fullPath)) {
                            $message->attach($fullPath, [
                                'as'   => basename($fullPath), // optional: rename in email
                                'mime' => mime_content_type($fullPath), // detect file type
                            ]);
                        }
                    }
                });
            }

            // ✅ Delete the image after sending to all subscribers
            if ($imageUrl) {
                $relativePath = str_replace(Storage::disk('public')->url(''), '', $imageUrl);
                if (Storage::disk('public')->exists($relativePath)) {
                    Storage::disk('public')->delete($relativePath);
                }
            }

            Notification::make()
                ->success()
                ->title(__("newsletter-subscribers-resource.send-success"))
                ->send();
        } catch (Exception $e) {
            Notification::make()
                ->danger()
                ->title(__("newsletter-subscribers-resource.send-failure"))
                ->body($e->getMessage())
                ->send();
        }
    }
}
