<?php

namespace App\Filament\Pages;

use App\Support\OurServicesPage as SupportPage;
use Exception;
use Filament\Forms\Form;
use Filament\Forms;
use Filament\Pages\Page;
use Illuminate\Contracts\Support\Htmlable;
use Filament\Actions\Action;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;

class OurServicesPage extends Page
{
    protected static ?string $navigationIcon = 'heroicon-o-sparkles';

    protected static string $view = 'filament.pages.our-services-page';

    public static function canAccess(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }

    public static function shouldRegisterNavigation(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }

    public array $data = [];

    public static function getNavigationGroup(): ?string
    {
        return __('navigation.cms');
    }

    public function getTitle(): string|Htmlable
    {
        return __('our-services.our-services');
    }

    public static function getNavigationLabel(): string
    {
        return __('our-services.our-services');
    }

    public function mount(): void
    {
        $page = SupportPage::make();

        $this->form->fill([
            'title' => $page->header()->getRaw('title'),
            'description' => $page->header()->getRaw('description'),
            'tag' => $page->header()->getRaw('tag'),
            'cover_title' => $page->header()->getRaw('cover_title'),
            /* 'VAT' => $page->header()->getRaw('VAT'),
            'discount_rate' => $page->header()->getRaw('discount_rate'),*/
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->statePath('data')
            ->schema([
                Forms\Components\Section::make(strip_tags(SupportPage::make()->header()->getRaw('title')[app()->getLocale()]))
                    ->description(strip_tags(SupportPage::make()->header()->getRaw('description')[app()->getLocale()]))
                    ->aside()
                    ->schema([

                        Forms\Components\SpatieMediaLibraryFileUpload::make('cover')
                            ->label(__('our-services.cover-label'))
                            ->collection(SupportPage::MEDIA_COLLECTION_COVER)
                            ->model(SupportPage::make()->model())
                            ->required(),
                        Forms\Components\RichEditor::make('title')
                            ->label(__('our-services.title-label'))
                            ->placeholder(__('our-services.title-placeholder'))
                            ->required()
                            ->translatable(),
                        Forms\Components\RichEditor::make('description')
                            ->label(__('our-services.description-label'))
                            ->placeholder(__('our-services.description-placeholder'))
                            ->required()
                            ->translatable(),
                        Forms\Components\RichEditor::make('tag')
                            ->label(__('our-services.tag-label'))
                            ->placeholder(__('our-services.tag-placeholder'))
                            ->required()
                            ->translatable(),
                        Forms\Components\RichEditor::make('cover_title')
                            ->label(__('our-services.cover-title-label'))
                            ->placeholder(__('our-services.cover-title-placeholder'))
                            ->required()
                            ->translatable(),
                        /* Forms\Components\Grid::make(2)
                            ->schema([
                                Forms\Components\TextInput::make('VAT')
                                    ->label(__('our-services.VAT-label'))
                                    ->placeholder(__('our-services.VAT-placeholder'))
                                    ->numeric()
                                    ->required(),

                                Forms\Components\TextInput::make('discount_rate')
                                    ->label(__('our-services.discount-rate-label'))
                                    ->prefix('%')
                                    ->placeholder(__('our-services.discount-rate-placeholder'))
                                    ->numeric()
                                    ->required(),
                            ]),*/
                    ]),
            ]);
    }

    protected function getActions(): array
    {
        return [
            Action::make(__('app.update'))
                ->icon('heroicon-o-arrow-path')
                ->action(fn() => $this->publish()),
        ];
    }

    public function publish(): void
    {
        $this->validate();

        try {
            DB::beginTransaction();

            $page = SupportPage::make();

            $image = collect($this->data['cover'])->first();
            if (! is_string($image) && TemporaryUploadedFile::class === get_class($image)) {
                $page->setCover($image);
            }

            $page->header()->patch([
                'title' => $this->data['title'],
                'description' => $this->data['description'],
                'tag' => $this->data['tag'],
                'cover_title' => $this->data['cover_title'],
                /*'VAT' => $this->data['VAT'],
                'discount_rate' => $this->data['discount_rate'],*/
            ]);

            Notification::make()
                ->success()
                ->title(__('app.content-update-success'))
                ->send();
            DB::commit();
        } catch (Exception $e) {
            DB::rollback();
            Log::error('Failed to update Our Services data', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            Notification::make()
                ->danger()
                ->title(__('app.content-update-failure'))
                ->send();
        }
    }
}
