<?php

namespace App\Filament\Pages;

use App\Support\OurPackagesPage;
use Exception;
use Filament\Forms\Form;
use Filament\Forms;
use Filament\Pages\Page;
use Illuminate\Contracts\Support\Htmlable;
use Filament\Actions\Action;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Rules\MaxPlainTextLength;

class OurPackages extends Page
{
    protected static ?string $navigationIcon = "heroicon-o-gift";

    protected static string $view = "filament.pages.our-packages";

        public static function canAccess(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }

    public static function shouldRegisterNavigation(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }

    public array $data = [];


    public static function getNavigationGroup(): ?string
    {
        return __("navigation.cms");
    }

    public function getTitle(): string|Htmlable
    {
        return __("our-packages.our-packages");
    }

    public static function getNavigationLabel(): string
    {
        return __("our-packages.our-packages");
    }

    public function mount(): void
    {
        $page = OurPackagesPage::make();

        $this->form->fill([
            'tag' => $page->header()->getRaw('tag'),
            'title' => $page->header()->getRaw('title'),
            'description' => $page->header()->getRaw('description'),
        ]);
    }

    public function form(Form $form): Form
    {
        return $form->statePath("data")->schema([
            Forms\Components\Section::make(strip_tags(OurPackagesPage::make()->header()->getRaw('title')[app()->getLocale()]))
                ->description(strip_tags(OurPackagesPage::make()->header()->getRaw('description')[app()->getLocale()]))
                ->aside()
                ->schema([
                    Forms\Components\SpatieMediaLibraryFileUpload::make('cover')
                        ->label(__('our-packages.cover-label'))
                        ->collection(OurPackagesPage::MEDIA_COLLECTION_COVER)
                        ->model(OurPackagesPage::make()->model())
                        ->required(),
                    Forms\Components\TextInput::make("tag")
                        ->label(__("our-packages.tag-label"))
                        ->placeholder(__("our-packages.tag-placeholder"))
                        ->required()
                        ->maxLength(255)
                        ->translatable(),
                    Forms\Components\TextInput::make("title")
                        ->label(__("our-packages.title-label"))
                        ->placeholder(__("our-packages.title-placeholder"))
                        ->required()
                        ->maxLength(15)
                        ->helperText(__('our-packages.title-helper'))
                        ->translatable(),
                    Forms\Components\RichEditor::make("description")
                        ->label(__("our-packages.description-label"))
                        ->placeholder(__("our-packages.description-placeholder"))
                        ->required()
                        ->translatable()
                ]),
        ]);
    }

    protected function getActions(): array
    {
        return [
            Action::make(__("app.update"))
                ->icon("heroicon-o-arrow-path")
                ->action(fn() => $this->publish()),
        ];
    }

    public function publish(): void
    {
        //$this->validate();
        $this->validate($this->getValidationRules());
        try {
            DB::beginTransaction();

            $page = OurPackagesPage::make();

            $image = collect($this->data['cover'])->first();
            if (! is_string($image) && TemporaryUploadedFile::class === get_class($image)) {
                $page->setCover($image);
            }

            $page->header()->patch([
                "tag" => $this->data["tag"],
                "title" => $this->data["title"],
                "description" => $this->data["description"],
            ]);

            Notification::make()
                ->success()
                ->title(__("app.content-update-success"))
                ->send();
            DB::commit();
        } catch (Exception $e) {
            DB::rollback();
            Log::error("Failed to update Our Packages data", [
                "message" => $e->getMessage(),
                "trace" => $e->getTraceAsString(),
            ]);

            Notification::make()
                ->danger()
                ->title(__("app.content-update-failure"))
                ->send();
        }
    }

    protected function getValidationRules(): array
    {
        $rules = [
            'data.tag' => ['required', 'array'],
            'data.title' => ['required', 'array'],
            'data.description' => ['required', 'array'],
            'data.cover' => ['required'],
        ];

        $plainTextLimits = [
            'title' => 15,
        ];

        foreach ($plainTextLimits as $field => $limit) {
            foreach ($this->data[$field] ?? [] as $locale => $_) {
                $rules["data.{$field}.{$locale}"][] = 'required';
                $rules["data.{$field}.{$locale}"][] = new MaxPlainTextLength($limit);
            }
        }

        return $rules;
    }
}
