<?php

namespace App\Filament\Pages;

use App\Support\NewsletterPage;
use Exception;
use Filament\Forms\Form;
use Filament\Forms;
use Filament\Pages\Page;
use Illuminate\Contracts\Support\Htmlable;
use Filament\Actions\Action;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class Newsletter extends Page
{
    protected static ?string $navigationIcon = "heroicon-o-newspaper";

    protected static string $view = "filament.pages.newsletter";

    public static function canAccess(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }

    public static function shouldRegisterNavigation(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }

    public array $data = [];


    public static function getNavigationGroup(): ?string
    {
        return __("navigation.cms");
    }

    public function getTitle(): string|Htmlable
    {
        return __("newsletter.newsletter");
    }

    public static function getNavigationLabel(): string
    {
        return __("newsletter.newsletter");
    }

    public function mount(): void
    {
        $page = NewsletterPage::make();

        $this->form->fill([
            'title' => $page->header()->getRaw('title'),
            'description' => $page->header()->getRaw('description'),
        ]);
    }

    public function form(Form $form): Form
    {
        return $form->statePath("data")->schema([
            Forms\Components\Section::make(NewsletterPage::make()->header()->getRaw('title')[app()->getLocale()])
                ->description(strip_tags(NewsletterPage::make()->header()->getRaw('description')[app()->getLocale()]))
                ->aside()
                ->schema([
                    Forms\Components\TextInput::make("title")
                        ->label(__("newsletter.title-label"))
                        ->placeholder(__("newsletter.title-placeholder"))
                        ->required()
                        ->maxLength(255)
                        ->translatable(),
                    Forms\Components\RichEditor::make("description")
                        ->label(__("newsletter.description-label"))
                        ->placeholder(__("newsletter.description-placeholder"))
                        ->required()
                        ->translatable()
                ]),
        ]);
    }

    protected function getActions(): array
    {
        return [
            Action::make(__("app.update"))
                ->icon("heroicon-o-arrow-path")
                ->action(fn() => $this->publish()),
        ];
    }

    public function publish(): void
    {
        $this->validate();

        try {
            DB::beginTransaction();

            $page = NewsletterPage::make();

            $page->header()->patch([
                "title" => $this->data["title"],
                "description" => $this->data["description"],
            ]);

            Notification::make()
                ->success()
                ->title(__("app.content-update-success"))
                ->send();
            DB::commit();
        } catch (Exception $e) {
            DB::rollback();
            Log::error("Failed to update Newsletter data", [
                "message" => $e->getMessage(),
                "trace" => $e->getTraceAsString(),
            ]);

            Notification::make()
                ->danger()
                ->title(__("app.content-update-failure"))
                ->send();
        }
    }
}
