<?php

namespace App\Filament\Pages;

use App\Support\HomePage;
use Exception;
use Filament\Forms\Form;
use Filament\Forms;
use Filament\Pages\Page;
use Illuminate\Contracts\Support\Htmlable;
use Filament\Actions\Action;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Rules\MaxPlainTextLength;

class Home extends Page
{
    protected static ?string $navigationIcon = "heroicon-o-home";

    protected static string $view = "filament.pages.home";

    public static function canAccess(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }

    public static function shouldRegisterNavigation(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }

    public array $data = [];


    public static function getNavigationGroup(): ?string
    {
        return __("navigation.cms");
    }

    public function getTitle(): string|Htmlable
    {
        return __("home.home");
    }

    public static function getNavigationLabel(): string
    {
        return __("home.home");
    }

    public function mount(): void
    {
        $page = HomePage::make();

        $this->form->fill([
            'title' => $page->header()->getRaw('title'),
            'description' => $page->header()->getRaw('description'),
        ]);
    }

    public function form(Form $form): Form
    {
        return $form->statePath("data")->schema([
            Forms\Components\Section::make(strip_tags(HomePage::make()->header()->getRaw('title')[app()->getLocale()]))
                ->description(strip_tags(HomePage::make()->header()->getRaw('description')[app()->getLocale()]))
                ->aside()
                ->schema([
                    Forms\Components\SpatieMediaLibraryFileUpload::make("cover")
                        ->label(__("home.cover-label"))
                        ->collection(HomePage::MEDIA_COLLECTION_COVER)
                        ->model(HomePage::make()->model())
                        ->required(),
                    Forms\Components\TextInput::make("title")
                        ->label(__("home.title-label"))
                        ->placeholder(__("home.title-placeholder"))
                        ->required()
                        ->maxLength(50)
                        ->helperText(__('home.title-helper'))
                        ->translatable(),
                    Forms\Components\RichEditor::make("description")
                        ->label(__("home.description-label"))
                        ->placeholder(__("home.description-placeholder"))
                        ->required()
                        ->maxLength(60)
                        ->helperText(__('home.description-helper'))
                        ->translatable()
                ]),
        ]);
    }

    protected function getActions(): array
    {
        return [
            Action::make(__("app.update"))
                ->icon("heroicon-o-arrow-path")
                ->action(fn() => $this->publish()),
        ];
    }

    public function publish(): void
    {
        // $this->validate();
        $this->validate($this->getValidationRules());
        try {
            DB::beginTransaction();

            $page = HomePage::make();

            $image = collect($this->data["cover"])->first();
            if (
                !is_string($image) &&
                TemporaryUploadedFile::class === get_class($image)
            ) {
                $page->setCover($image);
            }

            $page->header()->patch([
                "title" => $this->data["title"],
                "description" => $this->data["description"],
            ]);

            Notification::make()
                ->success()
                ->title(__("app.content-update-success"))
                ->send();
            DB::commit();
        } catch (Exception $e) {
            DB::rollback();
            Log::error("Failed to update Home data", [
                "message" => $e->getMessage(),
                "trace" => $e->getTraceAsString(),
            ]);

            Notification::make()
                ->danger()
                ->title(__("app.content-update-failure"))
                ->send();
        }
    }

    protected function getValidationRules(): array
    {
        $rules = [
            'data.title' => ['required', 'array'],
            'data.description' => ['required', 'array'],
            'data.cover' => ['required'],
        ];

        $plainTextLimits = [
            'title' => 50,
            'description' => 60
        ];

        foreach ($plainTextLimits as $field => $limit) {
            foreach ($this->data[$field] ?? [] as $locale => $_) {
                $rules["data.{$field}.{$locale}"][] = 'required';
                $rules["data.{$field}.{$locale}"][] = new MaxPlainTextLength($limit);
            }
        }

        return $rules;
    }
}
