<?php

namespace App\Filament\Pages;

use App\Enums\Role;
use App\Models\Setting;
use Filament\Facades\Filament;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Filament\Widgets\Widget;
use Filament\Widgets\WidgetConfiguration;
use Illuminate\Contracts\Support\Htmlable;

class Dashboard extends Page implements Forms\Contracts\HasForms
{
    use Forms\Concerns\InteractsWithForms;

   protected static ?string $navigationIcon = 'heroicon-o-home'; 
    protected static string $view = 'filament.pages.dashboard'; 
    protected static ?int $navigationSort = -2;
    protected static string $routePath = '/';

    public array $data = [];

    public static function getNavigationLabel(): string
    {
        return __('Dashboard');
    }

    public function mount(): void
    {
        $this->form->fill([
            'sales_email' => Setting::get('sales_email'),
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->statePath('data')
            ->schema([
                Forms\Components\Section::make('Sales Settings')
                    ->visible(fn () => auth()->user()?->hasRole(Role::admin->value) ?? false)
                    ->schema([
                        Forms\Components\TextInput::make('sales_email')
                            ->label('Sales Email')
                            ->email()
                            ->maxLength(191)
                            ->required(),
                        Forms\Components\Actions::make([
                            Forms\Components\Actions\Action::make('saveSalesEmail')
                                ->label('Save')
                                ->icon('heroicon-o-check')
                                ->action(fn () => $this->saveSalesEmail()),
                        ]),
                    ]),
            ]);
    }

    public function saveSalesEmail(): void
    {
        abort_unless(auth()->user()?->hasRole(Role::admin->value), 403);

        $this->validate([
            'data.sales_email' => ['required', 'email', 'max:191'],
        ]);

        $email = $this->form->getState()['sales_email'] ?? null;
        Setting::set('sales_email', $email);

        Notification::make()
            ->title('Sales email updated.')
            ->success()
            ->send();
    }

    public function getWidgets(): array
    {
        return Filament::getWidgets();
    }

    public function getVisibleWidgets(): array
    {
        return $this->filterVisibleWidgets($this->getWidgets());
    }
}
