<?php

namespace App\Filament\Pages;

use App\Support\ContactUsPage;
use Exception;
use Filament\Forms\Form;
use Filament\Forms;
use Filament\Pages\Page;
use Illuminate\Contracts\Support\Htmlable;
use Filament\Actions\Action;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;

class ContactUs extends Page
{
    protected static ?string $navigationIcon = 'heroicon-o-chat-bubble-bottom-center';

    protected static string $view = 'filament.pages.contact-us';

    public static function canAccess(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }

    public static function shouldRegisterNavigation(): bool
    {
        return auth()->user()?->hasRole(\App\Enums\Role::admin->value) ?? false;
    }


    public array $data = [];

    public static function getNavigationGroup(): ?string
    {
        return __('navigation.cms');
    }

    public function getTitle(): string|Htmlable
    {
        return __('contact-us.contact-us');
    }

    public static function getNavigationLabel(): string
    {
        return __('contact-us.contact-us');
    }

    public function mount(): void
    {
        $page = ContactUsPage::make();

        $this->form->fill([
            'description' => $page->header()->getRaw('description'),
            'message' => $page->header()->getRaw('message'),
            'email_subtitle' => $page->email()->getRaw('subtitle'),
            'email_description' => $page->email()->getRaw('description'),
            'phone_subtitle' => $page->phone()->getRaw('subtitle'),
            'phone_description' => $page->phone()->getRaw('description'),
            'whatsapp_subtitle' => $page->whatsapp()->getRaw('subtitle'),
            'whatsapp_description' => $page->whatsapp()->getRaw('description'),
            'address_description' => $page->address()->getRaw('description'),
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->statePath('data')
            ->schema([
                Forms\Components\Section::make(__('contact-us.header'))
                    ->description(strip_tags(ContactUsPage::make()->header()->getRaw('description')[app()->getLocale()]))
                    ->aside()
                    ->schema([
                        Forms\Components\SpatieMediaLibraryFileUpload::make('cover')
                            ->label(__('contact-us.cover-label'))
                            ->collection(ContactUsPage::MEDIA_COLLECTION_COVER)
                            ->model(ContactUsPage::make()->model())
                            ->required(),
                        Forms\Components\SpatieMediaLibraryFileUpload::make('logo')
                            ->label(__('contact-us.logo-label'))
                            ->collection(ContactUsPage::MEDIA_COLLECTION_LOGO)
                            ->model(ContactUsPage::make()->model())
                            ->required(),
                        Forms\Components\RichEditor::make('description')
                            ->label(__('contact-us.description-label'))
                            ->placeholder(__('contact-us.description-placeholder'))
                            ->required()
                            ->translatable(),
                        Forms\Components\RichEditor::make('message')
                            ->label(__('contact-us.message-label'))
                            ->placeholder(__('contact-us.message-placeholder'))
                            ->required()
                            ->translatable(),
                    ]),
                Forms\Components\Section::make(ContactUsPage::make()->email()->getRaw('subtitle')[app()->getLocale()])
                    ->description(ContactUsPage::make()->email()->getRaw('description'))
                    ->aside()
                    ->schema([
                        Forms\Components\TextInput::make('email_subtitle')
                            ->label(__('contact-us.subtitle-label'))
                            ->placeholder(__('contact-us.subtitle-placeholder'))
                            ->required()
                            ->maxLength(255)
                            ->translatable(),

                        Forms\Components\TextInput::make('email_description')
                            ->label(__('contact-us.description-label'))
                            ->placeholder(__('contact-us.description-placeholder'))
                            ->required(),

                        Forms\Components\SpatieMediaLibraryFileUpload::make('email_icon')
                            ->label(__('contact-us.email-icon-label'))
                            ->collection(ContactUsPage::MEDIA_COLLECTION_EMAIL_ICON)
                            ->model(ContactUsPage::make()->model())
                            ->required(),
                    ]),
                Forms\Components\Section::make(ContactUsPage::make()->phone()->getRaw('subtitle')[app()->getLocale()])
                    ->description(ContactUsPage::make()->phone()->getRaw('description'))
                    ->aside()
                    ->schema([
                        Forms\Components\TextInput::make('phone_subtitle')
                            ->label(__('contact-us.subtitle-label'))
                            ->placeholder(__('contact-us.subtitle-placeholder'))
                            ->required()
                            ->maxLength(255)
                            ->translatable(),

                        Forms\Components\TextInput::make('phone_description')
                            ->label(__('contact-us.description-label'))
                            ->placeholder(__('contact-us.description-placeholder'))
                            ->required(),

                        Forms\Components\SpatieMediaLibraryFileUpload::make('phone_icon')
                            ->label(__('contact-us.phone-icon-label'))
                            ->collection(ContactUsPage::MEDIA_COLLECTION_PHONE_ICON)
                            ->model(ContactUsPage::make()->model())
                            ->required(),
                    ]),
                Forms\Components\Section::make(ContactUsPage::make()->whatsapp()->getRaw('subtitle')[app()->getLocale()])
                    ->description(ContactUsPage::make()->whatsapp()->getRaw('description'))
                    ->aside()
                    ->schema([
                        Forms\Components\TextInput::make('whatsapp_subtitle')
                            ->label(__('contact-us.subtitle-label'))
                            ->placeholder(__('contact-us.subtitle-placeholder'))
                            ->required()
                            ->maxLength(255)
                            ->translatable(),

                        Forms\Components\TextInput::make('whatsapp_description')
                            ->label(__('contact-us.description-label'))
                            ->placeholder(__('contact-us.description-placeholder'))
                            ->required(),

                        Forms\Components\SpatieMediaLibraryFileUpload::make('whatsapp_icon')
                            ->label(__('contact-us.whatsapp-icon-label'))
                            ->collection(ContactUsPage::MEDIA_COLLECTION_WHATSAPP_ICON)
                            ->model(ContactUsPage::make()->model())
                            ->required(),
                    ]),
                Forms\Components\Section::make(__('contact-us.address'))
                    ->description(ContactUsPage::make()->address()->getRaw('description')[app()->getLocale()])
                    ->aside()
                    ->schema([
                        Forms\Components\TextInput::make('address_description')
                            ->label(__('contact-us.description-label'))
                            ->placeholder(__('contact-us.description-placeholder'))
                            ->required()
                            ->translatable(),

                        Forms\Components\SpatieMediaLibraryFileUpload::make('address_icon')
                            ->label(__('contact-us.address-icon-label'))
                            ->collection(ContactUsPage::MEDIA_COLLECTION_ADDRESS_ICON)
                            ->model(ContactUsPage::make()->model())
                            ->required(),
                    ]),
            ]);
    }

    protected function getActions(): array
    {
        return [
            Action::make(__('app.update'))
                ->icon('heroicon-o-arrow-path')
                ->action(fn() => $this->publish()),
        ];
    }

    public function publish(): void
    {
        $this->validate();

        try {
            DB::beginTransaction();

            $page = ContactUsPage::make();
            $image = collect($this->data['cover'])->first();

            if (! is_string($image) && TemporaryUploadedFile::class === get_class($image)) {
                $page->setCover($image);
            }

            $image = collect($this->data['logo'])->first();

            if (! is_string($image) && TemporaryUploadedFile::class === get_class($image)) {
                $page->setLogo($image);
            }

            $image = collect($this->data['email_icon'])->first();
            if (! is_string($image) && TemporaryUploadedFile::class === get_class($image)) {
                $page->setEmailIcon($image);
            }


            $image = collect($this->data['phone_icon'])->first();
            if (! is_string($image) && TemporaryUploadedFile::class === get_class($image)) {
                $page->setPhoneIcon($image);
            }


            $image = collect($this->data['whatsapp_icon'])->first();
            if (! is_string($image) && TemporaryUploadedFile::class === get_class($image)) {
                $page->setWhatsappIcon($image);
            }
            $image = collect($this->data['address_icon'])->first();
            if (! is_string($image) && TemporaryUploadedFile::class === get_class($image)) {
                $page->setAddressIcon($image);
            }

            $page->header()->patch([
                'description' => $this->data['description'],
                'message' => $this->data['message'],
            ]);

            $page->email()->patch([
                'subtitle' => $this->data['email_subtitle'],
                'description' => $this->data['email_description'],
            ]);
            $page->phone()->patch([
                'subtitle' => $this->data['phone_subtitle'],
                'description' => $this->data['phone_description'],
            ]);
            $page->whatsapp()->patch([
                'subtitle' => $this->data['whatsapp_subtitle'],
                'description' => $this->data['whatsapp_description'],
            ]);
            $page->address()->patch([
                'description' => $this->data['address_description'],
            ]);

            Notification::make()
                ->success()
                ->title(__('app.content-update-success'))
                ->send();
            DB::commit();
        } catch (Exception $e) {
            DB::rollback();
            Log::error('Failed to update Contact Us data', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            Notification::make()
                ->danger()
                ->title(__('app.content-update-failure'))
                ->send();
        }
    }
}
